clear,clc

% Given parameters
R=1; %m
g=9.82; %m/s^2
m=1; %kg
alpha=50*(pi/180); % cone angle in radians
v0=0.5*sqrt(g/tan(alpha)); %m/s
phi=45*(pi/180);  %angle between horizontal plane and v
mu=0; %coefficient of friction

% initial conditions
for i=1
    r0=R;
    r_dot0=-sin(phi)*v0*tan(alpha);
    
    theta_0=0;
    theta_dot0=cos(phi)*v0/R;
    
    z0=0;
    z_dot0=-(sin(phi)*v0);
end

% time integration parameters
dt=0.025;
tEnd=1000*dt;

%initialization
for i=1
    r=r0;
    r_dot=r_dot0;
    
    theta=theta_0;
    theta_dot=theta_dot0;
    
    z=z0;
    z_dot=z_dot0;
    
    rVec=[];
    thetaVec=[];
    zVec=[];
    timeVec=[];
    
    AngMomentum=[];
    TotalEnergy=[];
    vtotVec=[];
    diff_r=[];
    diff_theta=[];
    
    fig=figure('Position',[100 100 600 600]);
    set(gcf,'currentchar','') % set dummy character to allow for key-press abortion of simulation
    
    stopCondition=0;
    t=0;
    u=[0 0 0 0];
end
while stopCondition==0 && get(gcf,'currentchar')==''
    % total velocity
    v=sqrt(r_dot^2+r^2*theta_dot^2+z_dot^2);
    
    % store data
    rVec=[rVec;r];
    thetaVec=[thetaVec;theta];
    zVec=[zVec;z];
    
    % Calculate Angular momentum
    AngMomentum=[AngMomentum; m*r^2*theta_dot];
    
    % Calculate Total energy
    TotalEnergy=[TotalEnergy; m*v^2/2+m*g*z];
    
    vtotVec=[vtotVec;v];
    timeVec=[timeVec;t];
    
    % Calculate normal force
    N=(m*r*theta_dot^2+m*g*tan(alpha))/(sin(alpha)*tan(alpha)+cos(alpha));
    
    % define tangential direction in cyl.coord.
    e_t=[r_dot r*theta_dot z_dot]/v;
    
    % define friction force in cyl.coord.
    f_r=-mu*N*e_t(1);
    f_theta=-mu*N*e_t(2);
    f_z=-mu*N*e_t(3);
    
    %% calculate the instantanous acceleration
    u(1)=r;
    u(2)=theta;
    u(3)=r_dot;
    u(4)=theta_dot;
    
    % send data to runge-kutta 4
    info=[N m alpha g f_r f_theta f_z];
    u=rungekutt4(u,info,dt);
    
    % update position and velocities
    r=u(1);
    r_dot=u(3);
    
    theta=u(2);
    theta_dot=u(4);
    
    z=(r-R)/tan(alpha);
    z_dot=r_dot/(tan(alpha));
    
    
    % Stop simulation if bead is too close to the bottom of the cone
    if z<-R/tan(alpha)*0.9
        stopCondition=1;
    end
    
    % Plot the trajectory
    xMax=R*2;
    yMax=R*2;
    zMin=-R/tan(alpha)*1.2;
    zMax=R/tan(alpha)*0.5;
    
    figure(fig)
    plot3(rVec.*cos(thetaVec),rVec.*sin(thetaVec),zVec,'LineStyle','-','Color','b','LineWidth',2)
    line(rVec.*cos(thetaVec),yMax*ones(size(thetaVec)),zVec,'LineStyle','-','Color',[0.5 0.5 1])
    line(xMax*ones(size(thetaVec)),rVec.*sin(thetaVec),zVec,'LineStyle','-','Color',[0.5 0.5 1])
    line(rVec.*cos(thetaVec),rVec.*sin(thetaVec),zMin*ones(size(thetaVec)),'LineStyle','-','Color',[0.5 0.5 1])
    line(rVec(end).*cos(thetaVec(end)),rVec(end).*sin(thetaVec(end)),zVec(end),'LineStyle','none','Marker','o','MarkerSize',10,'MarkerFaceColor','b','MarkerEdgeColor','k','LineWidth',2)
    line([0 R],yMax*ones(size([0 R])),[-R/tan(alpha) 0],'LineWidth',2,'Color',[0.2 0.2 0.2])
    line([0 -R],yMax*ones(size([0 R])),[-R/tan(alpha) 0],'LineWidth',2,'Color',[0.2 0.2 0.2])
    line(xMax*ones(size([0 R])),[0 R],[-R/tan(alpha) 0],'LineWidth',2,'Color',[0.2 0.2 0.2])
    line(xMax*ones(size([0 R])),[0 -R],[-R/tan(alpha) 0],'LineWidth',2,'Color',[0.2 0.2 0.2])
    axis equal
    
    xlim([-xMax xMax])
    ylim([-xMax xMax])
    zlim([zMin zMax])
    
    set(gca,'XTick',[-R 0 R],'YTick',[-R 0 R],'ZTick',[-R/tan(alpha) 0])
    grid on
    
    title('Press button to stop simulation')
    
    drawnow
    
    t=t+dt;
    
    % Energy and angular momentum correction for conservative system
    if mu==0
        E=TotalEnergy(1);
        Hz=AngMomentum(1);
        
        r_dot_true=sign(r_dot)*sqrt(((2*(E-m*g*z)/m)-(Hz^2/(m*r)^2))/(1+tan(alpha)^(-2)));
        theta_dot_true=Hz/(m*r^2);
        z_dot_true=r_dot_true/(tan(alpha));
        
        r_dot=r_dot_true;
        theta_dot=theta_dot_true;
        z_dot=z_dot_true;
    end
    
    
end
figStatus=figure('Position',[800 100 600 600]);
subplot(2,1,1)
plot(timeVec,m*vtotVec.^2/2,'Color','r')
line(timeVec,m*g*(zVec+R/tan(alpha)),'Color','b')
line(timeVec,m*vtotVec.^2/2+m*g*(zVec+R/tan(alpha)),'Color','k','LineWidth',2)

xlabel('Time [s]')
ylabel('Energy [J]')
legend('Kinetic','Potential','Total')

subplot(2,1,2)
plot(timeVec,AngMomentum)
xlabel('Time [s]')
ylabel('Angular momentum around z-axis [Js]')

figure(fig)
FileName=['Bead_in_cone_alpha' num2str(alpha*(180/pi)) '_v' num2str(v0) '_mu' num2str(mu)];
set(gcf,'PaperUnits','inches','PaperType','usletter', 'PaperPosition', [0 0 16 12]);
saveas(gcf,[FileName '.png'],'png')
saveas(gcf,[FileName '.eps'],'epsc')