%--------------------------------------------------------------------------
%
% Kuramoto Sivashinsky equation
%
% LQG compensator.
%
% A LQG compensator is designed coupling a LQR controller and a Kalman
% filter. The compensator is tested when the system is excited by a white
% Gaussian noise d(t).
% Saturation and actuator displecemnt are implemented: to activate
% uncomment the respective code lines.
%
% Reference figure(s): Fig. 25, Fig. 26, Fig. 27
% 
% Nicolo' Fabbiane, December 2013
% KTH Mechanics
% nicolo@mech.kth.se
%
%--------------------------------------------------------------------------
clc, clear all
fprintf('\nLQG compensator.\n\n')




%% Initialization of the plant --------------------------------------------

% LTI system matrix A and x grid
nq = 400; % number of grid points

[A,x,I] = KS_init(nq);


% Inputs matrix B

% disturbance d (Gaussian shape at x_d with sigma_d variance)
x_d = 35; sigma_d = 4; 
Bd  = exp(-((x-x_d)/sigma_d).^2)/sigma_d;

% actuator u (Gaussian shape at x_u with sigma_u variance)
x_u = 400; sigma_u = 4; 
Bu  = exp(-((x-x_u)/sigma_u).^2)/sigma_u;


% Outputs matrix C

% measurement y (Gaussian shape at x_y with sigma_y variance)
x_y = 300; sigma_y = 4; 
Cy  = (exp(-((x-x_y)/sigma_y).^2)/sigma_y).' * I;

% output z (Gaussian shape at x_z with sigma_z variance)
x_z = 700; sigma_z = 4; 
Cz  = (exp(-((x-x_z)/sigma_z).^2)/sigma_z).' * I;




%% LQR controller ---------------------------------------------------------

Wz = 1.0e+0; % z-output weight
Wu = 1.0e+0; % control penalty

% Solution of the control Riccati equation (Eqn. 66)
X = care(A,Bu,Cz'*Wz*Cz,Wu);

% Control gains matrix K (Eqn. 65)
K = - Wu\Bu'*X;




%% Kalman filter ----------------------------------------------------------

Rd = 1.0e+0; % d variance
Rn = 1.0e-2; % n variance

% Solution of the estimation Riccati equation (Eqn. 91)
Re = care(A',Cy',Bd*Rd*Bd',Rn);

% Control gains matrix K (Eqn. 90)
L = - Re*Cy'/Rn;




%% Actuator displacement (uncomment to activate) --------------------------
% x_u = x_u + 5; 
% Bu = exp(-((x-x_u)/sigma_u).^2)/sigma_u;

        

        
%% Time integration -------------------------------------------------------

% Parameters
dt   = 1;
Tend = 50000;

Ton = 4000;

t  = 0:dt:Tend;
nt = length(t);


% Time-stepper matrices (Crank-Nicholson)
CNi = (speye(nq) - dt/2*A);
CNe = (speye(nq) + dt/2*A);


% Noises inizialization
d = randn(1,nt);     % unitary variance Gaussian white noise
d = d - mean(d,2);   % enforce zero-mean
d = d / std(d,[],2); % enforce unitary variance

n = randn(1,nt);     % unitary variance Gaussian white noise
n = n - mean(n,2);   % enforce zero-mean
n = n./ std(n,[],2) * 1e-1;


% Variables initialization
v = zeros(nq,nt); % velocity
y = zeros(1 ,nt); % measurement signal
z = zeros(1 ,nt); % output signal
u = zeros(1 ,nt); % control signal

v_cmp = zeros(nq,nt); % compensator state
y_cmp = zeros(1 ,nt); % estimated measurement signal


% Time-loop
for i = 1:nt-1
    
    % outputs
    y(:,i) = Cy*v(:,i) + n(:,i);
    z(:,i) = Cz*v(:,i);
    
    % control forcing (Eqn. 109)
    u(:,i) = K*v_cmp(:,i);
    
    % saturation (uncomment to activate)
    % u(:,i) = min([max([-2,u(:,i)]),2]);
    
    % timestep
    v(:,i+1) = CNi\(CNe*v(:,i) + dt * Bd*d(:,i) + dt * Bu*u(:,i));
    
    % LQG compensator
    if t(i) >= Ton
    % - estimated measurement
        y_cmp(:,i) = Cy*v_cmp(:,i);
    % - timestep
        v_cmp(:,i+1) = CNi\(CNe*v_cmp(:,i) + dt * Bu*u(:,i) ...
                                           + dt * L *(y_cmp(:,i)-y(:,i)));
    end
    
    % runtime output
    runtime_output

end




%% Velocity statistics ----------------------------------------------------
Tsta = 20000;
ista = (Tend-Tsta <= t) & (t <= Tend);

% Root-Mean-Square
v_rms = sqrt( mean(v(:,ista).^2,2) - mean(v(:,ista),2).^2 );

% visualization
rms_style  = '-r';
rms_legend = 'LQG';

plot_rms




%% Figure 25 --------------------------------------------------------------
figure(25); clf

subplot(8,1,2:3); imagesc(t,x,v); hold on
                  plot([0 Tend],[1 1]*x_y,'-k',...
                       [0 Tend],[1 1]*x_z,'-k','Linewidth',2);
                  plot([Ton Ton], [0 800],'--b','Linewidth',2); hold off
                  axis xy; axis([0 10000 0 800]);
                  set(gca,'XTickLabel',[]); ylabel('x')
                  
                  caxis([-1 1]*5); cax = caxis;
                  colormap(jet(512));
                    
subplot(8,1,1);   caxis(cax); colormap(jet(512));
                  pos = get(gca,'Position'); pos(4) = pos(4)/3;
                  set(gca,'Visible','off','Position',pos)
                  hc = colorbar('NO'); xlabel(hc,'v(x,t)')
                  set(hc,'XTick',cax(1):1:cax(2),...
                         'XTickLabel',cax(1):1:cax(2),'Position',pos);

subplot(8,1,4:5); imagesc(t,x,v_cmp); hold on
                  plot([0 Tend],[1 1]*x_y,'-k',...
                       [0 Tend],[1 1]*x_z,'-k','Linewidth',2);
                  plot([Ton Ton], [0 800],'--b','Linewidth',2); hold off
                  axis xy; axis([0 10000 0 800]);
                  set(gca,'XTickLabel',[]); ylabel('x')
                  
                  caxis(cax); colormap(jet(512));
                  
subplot(8,1,6);   plot(t,y,'-k','LineWidth',1); hold on
                  plot([Ton Ton],[-10 10],'--b','Linewidth',2); hold on
                  axis([0 10000 -10 10]);
                  set(gca,'XTickLabel',[]); ylabel('y(t)'), grid on
                  
subplot(8,1,7);   plot(t,u,'-k','LineWidth',1); hold on
                  plot([Ton Ton],[-5 5],'--b','Linewidth',2); hold on
                  axis([0 10000 -5 5]);
                  set(gca,'XTickLabel',[]); ylabel('u(t)'), grid on
                  
subplot(8,1,8);   plot(t,z,'-k','LineWidth',1); hold on
                  plot([Ton Ton],[-50 50],'--b','Linewidth',2); hold on
                  axis([0 10000 -50 50]);
                  xlabel('t'), ylabel('z(t)'), grid on