close all; clear; set(0,'DefaultLineLineWidth',0.5,'DefaultLineMarkerSize',2);
%
%---------------------------------------------------------------------------------------
% Global spectrum of the subcritical  Ginzburg-Landau equation :
%
%  All the eigenvalues (blue dots) are in the stable half-plane. The unstable
%  domain is in gray and the exact global spectrum is indicated in green. The
%  numerically computed global eigenvalues (blue dots) exhibit a characteristic
%  split, aligning with the resolvent contour that approximately represents
%  machine precision. The resolvent norm contours range from 10^{-1} to
%  10^{15}.
%
%
%  Shervin Bagheri, 16 March 2008
%  KTH Mechanics
%  shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------
%

%
%  Parameters of the supcritical Ginzburg-Landau equation
%
nx = 220;     
[xh,A,Q,cd,U,mu0,mu2,cc,gam] = setup_subcritical(nx);
  
%  
% Compute the eigenvalues 
%
Lam = eig(A); 
h = sqrt(-2*mu2*gam);
UU = U+2*sqrt(-1)*cc;
nn = [0:nx+1];
LamExact = mu0-cc^2 - (UU^2/(4*gam)) - (nn+0.5)*h;

%
% Frequency domain (Increase nres for higher resolution of spectrum)
%
nres = 50;
xmin = -10; xmax = 2;
ymin = -4; ymax = 12;
 
%
% Unstable half plane
%
xun = [0 xmax xmax 0 0];
yun = [ymin ymin ymax ymax ymin];  

%
%  Compute the resolvent norm
%
for i=1:nres
  for j=1:nres 
    lax = xmin + (xmax-xmin)/(nres-1)*(i-1);    
    lay = ymin + (ymax-ymin)/(nres-1)*(j-1);
    la = lax + sqrt(-1)*lay;    
    RR(i,j) = log10(norm(inv(la*eye(nx)-A)));;
  end
end

%
% Plot the pseudospectrum (Fig 10)
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 8]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);
xx = linspace(xmin,xmax,nres);
yy = linspace(ymin,ymax,nres);
fill(xun,yun,[0.66 0.66 0.66]); hold on;
RRmax = max(max(RR)); RRmin = min(min(RR));
contour(xx,yy,transpose(RR),[round(RRmin):1:round(RRmax)-3],'k');
plot(real(LamExact),imag(LamExact),'g.','MarkerSize',8)
plot(real(Lam),imag(Lam),'b.','MarkerSize',8); hold off; 
axis([xmin xmax ymin ymax]); axis square 
setlabel('\lambda_r','\lambda_i'); 
