close all; clear; set(0,'DefaultLineLineWidth',0.5,'DefaultLineMarkerSize',7);
%
%---------------------------------------------------------------------------------------
%   Example of the input-output behavior of the Ginzburg-Landau equation:
%
%  (a) The evolution in space and time of the state when forced by random noise
%  at $x_w=-11$. The region between the dashed lines is convectively unstable.
%
%  (b) Input signal (white noise) at $x_w=-11$, marked by an arrow in (a). The
%  amplitude of the input is less than 10.
%
%  (c) Output signal at $x_{s,1}=-8.2$ (branch I). The amplitude of the state
%  $y_1=C_1q$ has decreased to less than one.
%
%  (d) Output signal at $x_{s,2}=8.2$ (branch II). The signal has been amplified
%  by a factor of $20.$%
%
%  Shervin Bagheri, 16 March 2008
%  KTH Mechanics
%  shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------
%

%
%  Parameters of the  subcritical d Ginzburg-Landau equation
%
nx = 220;
[xh,A,Q,cd,U,mu0,mu2,cc,gam] = setup_subcritical(nx);

%
% Unstable domain: branch I and II
%
x1=-sqrt(-2*(mu0-cc^2)/mu2);
x2=sqrt(-2*(mu0-cc^2)/mu2);

% 
% Input B at x=-11
%
xw = -11; 
var = 0.4;
B1 = exp(-((xh-xw)/var).^2);

%
% Output  C1 at branch I 
%
xs = x1;
w = diag(Q);
C1 = exp(-((xh-xs)/var).^2).*w; C1 = C1';

%
% Output C2 at branch I I
%
xj = x2;
C2 = exp(-((xh-xj)/var).^2).*w; C2 = C2';  

%
% Time vector
%
tmax = 400; dt = 0.1;
t = 0:dt:tmax; 
nt = length(t);

%
% Initialize state vector
%
q   = zeros(nx,nt);

%
% March in time (Cranck Nicholson scheme)
%
I = eye(nx);
Mm = (I-dt/2*A); invMm = inv(Mm);
Mp = (I+dt/2*A);

%
% White noise
%
wh = randn(1,length(t))/sqrt(dt);

for tind=1:length(t)-1
  E(tind) = real(diag(q(:,tind)'*Q*q(:,tind)));
  q(:,tind+1) = invMm*(Mp*q(:,tind)+dt/2*B1*(wh(tind+1)+wh(tind)));
end

%
% Plot spatio-temporal evolution of response (Fig 11)
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 16 6]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);
subplot(3,2,[1,3,5])
MM=60;
surf(t,xh(MM:end-MM),real(q(MM:end-MM,:)));
ccolor; hold on;

%
% Find and plot branch I and II
%
dd(1)=find(xh>xs,1,'first');
dd(2)=find(xh>xj,1,'first');
vvv=dd(1):dd(2);
plot3(t,xh(dd(1))*ones(length(t),1),real(q(dd(1),:)),'k--')
plot3(t,xh(dd(2))*ones(length(t),1),real(q(dd(2),:)),'k--')
view(2); xlim([0 400]);  axis tight; setlabel('t','x')
annotation1 = annotation(fig1,'arrow',[0.46 0.46],[0.64 0.64]);
annotation5 = annotation(fig1,'arrow',[0.46 0.46],[0.3764 0.3764]);
annotation6 = annotation(fig1,'arrow',[0.46 0.46],[0.4125 0.4125]);
annotation2 = annotation( fig1,'textbox', 'Position',[0.46 0.41 0.06875 0.07111], 'LineStyle','none','Color',[0 0 0], 'FitHeightToText','off', 'FontName','times new roman', 'FontSize',7, 'String',{'B'});
annotation3 = annotation( fig1,'textbox', 'Position',[0.46 0.6422 0.06875 0.07111],  'LineStyle','none', 'Color',[1 0 0], 'FitHeightToText','off',  'FontName','times new roman',  'FontSize',7,  'String',{'C_2'});
annotation4 = annotation(  fig1,'textbox',  'Position',[0.46 0.44 0.06875 0.07111],  'LineStyle','none', 'Color',[1 0 0], 'FitHeightToText','off',  'FontName','times new roman',  'FontSize',7,  'String',{'C_1'});

%
% Plot input/output signals
%
subplot(3,2,2)
plot(t,real(C2*q),'b');
subplot(3,2,4)
plot(t,real(C1*q),'r');
subplot(3,2,6)
plot(t,wh,'k'); setlabel('t','')
