close all; clear; set(0,'DefaultLineLineWidth',2,'DefaultLineMarkerSize',7);
%
%---------------------------------------------------------------------------------------
%
%  (a) Input-output pseudospectra where the black transfer function contour
%  levels are $\{10^1, 10^2,\dots,10^6\}$. The red contour (with level $208$)
%  represents the largest contour value that crosses the imaginary axis. The
%  blue symbols indicate the eigenvalues of $A.$
%
%  (a) The state response to harmonic forcing located upstream of branch I
%  (lower of the two dashed lines). The largest response is at branch II (upper
%  dashed line) for $\omega=-0.65$.
%
%  (b) The frequency response, where the output is a Gaussian function (see
%  appendix~\ref{sec:disc}) located at branch II. In the gray area all forcing
%  frequencies are amplified in the unstable domain, all other frequencies are
%  damped illustrating a filtering effect. This response corresponds to the
%  thick dashed line representing the imaginary axis in the pseudospectra plot
%  in Figure~\ref{response:harm a}, and the peak value $\|G\|_\infty=208$
%  corresponds to the red contour level.
%
%  Shervin Bagheri, 16 March 2008
%  KTH Mechanics
%  shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------
%

%
%  Parameters of the subcritical Ginzburg-Landau equation
%
nx = 220;
[xh,A,M,cd,U,mu0,mu2,cc,gam] = setup_subcritical(nx);

%
% Unstable domain: branch I and II
%
x1=-sqrt(-2*(mu0-cc^2)/mu2);
x2=sqrt(-2*(mu0-cc^2)/mu2);

% 
% Input B at branch I
%
xw = x1; 
var =0.4;
B = exp(-((xh-xw)/var).^2);

% 
% Output C is factored energy C'*C=M
%
C = sqrth(M);

%
% Frequency response
%
lax = linspace(-1.,1.,200);
no = length(lax);
R = zeros(nx,no);

for ind = 1:no
    la=sqrt(-1)*lax(ind);
    R(:,ind)=inv(la*eye(nx)-A)*B;
    R2(ind)=max(svd(C*inv(la*eye(nx)-A)*B));
end

%
% Pseudospectrum of I/O
%
nres=50;
xmin = -4; xmax = 0.5;
ymin = -1.; ymax = 1;
xxx = linspace(xmin,xmax,nres); 
yyy = linspace(ymin,ymax,nres);
RR = zeros(nres,nres);
for ind=1:nres
  disp(['computing the pseudospectra:' num2str(ind)]);
  for jnd=1:nres
    laxs = xmin + (xmax-xmin)/(nres-1)*(ind-1);
    lay = ymin + (ymax-ymin)/(nres-1)*(jnd-1);
    la=laxs+sqrt(-1)*lay;
    RR(ind,jnd)=(max(svd(C*inv(la*eye(nx)-A)*B)));
  end
end

%
% Spectrum of A
%
[tt,ttt] = oeig(A);

%
% Plot Pseudospectrum (Fig 13)
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 8]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);
maxR=max(R2);
fill([0 0 xmax xmax 0],[ymin ymax ymax ymin ymin],[0.9 0.9 0.9],'EdgeColor','none')
hold on;
plot(real(ttt),imag(ttt),'bo','MarkerFaceColor',[0 0 1],'MarkerSize',2);
[C1,h]=contour(xxx,yyy,transpose(RR),[1 10 maxR ],'r','LineWidth',1);
[C1,h]=contour(xxx,yyy,transpose(RR),[1 10 1000 1e4 1e5 1e6],'k','LineWidth',1);
xlim([xmin  xmax]); ylim([ymin ymax]); 
setlabel('s_i','s_r')
plot(zeros(length(yyy),1),yyy,'k--')
hold off


%
% Plot Frequency response (B-->I) (Fig 14)
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 8]);
subplot(8,1,[1:5])
MM=70;
surf(lax,xh(MM:end-MM),real(R(MM:end-MM,:)));
ccolor; 
axis([-1.2 1.2 -100 100]); hold on;

%
% Find and plot unstable domain
%
dd(1)=find(xh>x1,1,'first');
dd(2)=find(xh>x2,1,'first');
vvv=dd(1):dd(2);
plot3(lax,xh(dd(1))*ones(length(lax),1),real(R(dd(1),:)),'k--','linewidth',1)
plot3(lax,xh(dd(2))*ones(length(lax),1),real(R(dd(2),:)),'k--','linewidth',1)
setlabel('','x');  axis tight;  view(2);

%
% Plot Frequency response (B-->C) 
%
subplot(8,1,[6:8])
ymin = 1e-5; ymax = maxR+10;
ee=find(abs(R2)>1,1,'last');
fill([lax(1:ee) lax(ee) 0] ,[abs(R2(1:ee)) 1e-5 1e-5],[0.9 0.9 0.9],'edgecolor',[0.9 0.9 0.9])
hold on;
plot(lax,abs(R2),'r',lax,ones(no,1),'k:','LineWidth',1)
axis([-1. 1. ymin-0.2 ymax+0.2]); set(gca,'XTick',-1.:0.25:1.)
setlabel('\omega','|G|'); hold off


