close all;clear; set(0,'DefaultLineLineWidth',2,'DefaultLineMarkerSize',7);

%
%---------------------------------------------------------------------------------------
%
% The response to stochastic forcing. (a) The evolution of the state energy
% for five different simulations (black lines), the mean state energy given
% by the solution of the algebraic Lyapunov equation (red solid line) and
% the energy averaged over 50 simulations (thick dashed line). (b) The thick
% red line shows the rms-value of the \GL\ when excited by random forcing
% $w$ at the location marked with an arrow. Five representative snapshots of
% the response to this forcing are shown by black thin lines; the average
% over 50 simulations is displayed by a thick blue dashed line.  
%
%
% Shervin Bagheri, 16 March 2008
% KTH Mechanics
% shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------

%
%  Parameters of the subcritical Ginzburg-Landau equation
%
nx = 220;
[xh,A,Q,cd,U,mu0,mu2,cc,gam] = setup_subcritical(nx);

%
% Input B1 at x=-11
% Covariance W=1
%
var = 0.4;
xw = -11; 
B1 = exp(-((xh-xw)/var).^2);
W = 1;

%
% Adjoint matrices
%
Ap = Q\A'*Q;
B1p = B1'*Q;

%
% State covarice when t--> infinite
%
V = lyap(A,Ap,B1*W*B1p);
mean = trace(V);
rms = sqrt(real(diag(V)));

%
% Time vector
%
tmax = 500;
dt = 0.5; tvec = 0:dt:tmax;

%
% Unstable domain: branch I and II
%
x1 = -sqrt(-2*(mu0-cc^2)/mu2);
x2 = sqrt(-2*(mu0-cc^2)/mu2);


%
% Perform 50 realization by marching in time (Cranck Nicholson)
%
I=eye(nx);
Mm=(I-dt/2*A);invMm=inv(Mm);
Mp=(I+dt/2*A);

ymax = 10;
L = 20;
qavg = zeros(nx,1);

for ind=1:50
  disp(['realization' num2str(ind)])
  
  %
  % White noise
  %
  wh=randn(1,length(tvec))/sqrt(dt);
  
  %
  % Initialize state vector
  %
  q=zeros(nx,length(tvec));

  tind2 = 1;
  for tind=1:length(tvec)-1;
    E(tind) = real(diag(q(:,tind)'*Q*q(:,tind)));
    
    q(:,tind+1)=invMm*(Mp*q(:,tind)+dt/2*B1*(wh(tind+1)+wh(tind)));
    
    %
    % Plot continuously the statistics
    %
    if tind2==tind
      subplot(2,1,1);
      fill([x1 x2 x2 x1 x1],[0 0 ymax ymax 0],[0.66 0.66 0.66]);
      hold on;
      fill([xw-2*var xw+2*var xw+2*var xw-2*var xw-2*var],[0 0 ymax ymax 0],[0.1 0.1 0.1]);
      plot(xh,abs(q(:,tind+1)),'k',xh,rms,'r--');
      axis([-L L 0 ymax])
      setlabel('x','Disturbance'); drawnow; hold off;
      
      subplot(2,1,2);
      semilogy(tvec,real(mean)*ones(length(tvec),1),'r--'); 
      hold on;
      plot(tvec(1:tind),E(1:tind),'k')
      hold off;
      setlabel('time','Energy');     drawnow;
      
      tind2 = tind + 10;
    end 
      
    %
    % Collect snapshot at end time to average
    %
    if tind==500
      qavg = qavg+abs(q(:,500));
    end

  end
  
  E(tind+1)=real(diag(q(:,tind+1)'*Q*q(:,tind+1)));
  
  Etot(ind,:)=E;
  
  %
  % Save 5 first simulations for plotting
  %
  if ind==1 ;     q1 = q; end 
  if ind==2 ;     q2 = q; end 
  if ind==3 ;     q3 = q; end 
  if ind==4 ;     q4 = q; end 
  if ind==5 ;     q5 = q; end 
 
end

%
% Average energy and state
%
meandeterm = sum(Etot,1)/ind;
qavg = qavg/ind;

%
% Plot state vectors and rms (Fig 15)
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 10]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);
subplot(2,1,1)
range= 1:5;
fill([x1 x2 x2 x1 x1],[0 0 ymax ymax 0],[0.9 0.9 0.9],'EdgeColor',[0.9 0.9 0.9]);
hold on;
plot(xh,abs(q1(:,500)),'k',xh,abs(q2(:,500)),'k',xh,abs(q3(:,500)),'k',xh,abs(q4(:,500)),'k',xh,abs(q5(:,500)),'k','Linewidth',0.5)
plot(xh,qavg,'b--','LineWidth',2)
plot(xh,rms,'r')
axis([-L L 0 9])
setlabel('x','');
hold off;

%
% Plot energy and mean energy
%
subplot(2,1,2)
semilogy(tvec,Etot(range,:),'k','LineWidth',0.5)
hold on;
plot(tvec,real(mean)*ones(length(tvec),1),'r')
plot(tvec,meandeterm,'b--','LineWidth',2)
hold off;
setlabel('t','');
ylim([1e-2 9e3])


    