close all;clear; set(0,'DefaultLineLineWidth',1,'DefaultLineMarkerSize',7);
%
%---------------------------------------------------------------------------------------
%
%  Controllability and observability Gramians
%
%  The state covariance/controllability Gramian $P$ of the Ginzburg-Landau
%  equation. The Gramian describing how the state components are influenced by
%  an input corresponds in a stochastic framework to the state covariance for
%  white noise as input. The red circle signifies the forcing location
%  ($x_w=-11$), the dashed box marks the region of instability. The states that
%  are most sensitive to forcing, and thus controllable, are located downstream,
%  at branch II.
%
% The observability Gramian $Q$ of the Ginzburg-Landau equation. The red circle
% markes the location of the output $C$ at branch II. The initial states that
% contribute most to the output are located upstream, at branch I.
%
%  Shervin Bagheri, 16 March 2008
%  KTH Mechanics
%  shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------
%

%
%  Parameters of the subcritical Ginzburg-Landau equation
%
nx = 220; 
[xh,A,Q,cd,U,mu0,mu2,cc,gam] = setup_subcritical(nx);



%
% Input B at x=-11
% with covariance W= 1.
%
var = 0.4;
xw = -11; 
B1 = exp(-((xh-xw)/var).^2);
W=1;		

%
% Solve algebraic lyapunov equation
%
V = lyap(A,B1*W*B1');
mean = trace(V*Q);
urms = sqrt(real(diag(V)));

%
% Output C at x=9
%
w = diag(Q);
xj = 9;
C1 = exp(-((xh-xj)/var).^2);C1=C1';
W=1;		
Vc = lyap(A',C1'*W*C1);
urms2 = sqrt(real(diag(Vc)));

%
% Unstable domain: branch I and II
%
x1=-sqrt(-2*mu0/mu2);
x2=sqrt(-2*mu0/mu2);

%
% Plot Controllability Gramian / state covariance (Fig 16)
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 8]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);
MM=70;
surf(xh(MM:end-MM),xh(MM:end-MM),real(V(MM:end-MM,MM:end-MM)));
ccolor;
hold on;

%
% Find and plot unstable domain
%
dd(1)=find(xh>x1,1,'first');
dd(2)=find(xh>x2,1,'first');
vvv=dd(1):dd(2);
plot3(xh(vvv),ones(length(vvv),1)*xh(dd(1)),real(V(vvv,dd(1))),'k--','linewidth',1)
plot3(xh(vvv),ones(length(vvv),1)*xh(dd(2)),real(V(vvv,dd(2))),'k--','linewidth',1)
plot3(ones(length(vvv),1)*xh(dd(1)),xh(vvv),real(V(vvv,dd(1))),'k--','linewidth',1)
plot3(ones(length(vvv),1)*xh(dd(2)),xh(vvv),real(V(vvv,dd(2))),'k--','linewidth',1)
view(2);  axis tight

%
% Mark forcing location with red circle
%
vv=find(xh>xw,1,'first');
plot3(xh(vv),xh(vv),real(V(vv,vv)),'ro','Markersize',4,'MarkerFaceColor',[1 0 0])
setlabel('x','x')

%
% Plot Observability Gramian / adjoint state covariance (Fig 18)
%
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 8]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);
MM=70;
surf(xh(MM:end-MM),xh(MM:end-MM),real(Vc(MM:end-MM,MM:end-MM)));
ccolor;
hold on;

%
% Plot unstable domain
%
vvv=dd(1):dd(2);
plot3(xh(vvv),ones(length(vvv),1)*xh(dd(1)),real(Vc(vvv,dd(1))),'k--','linewidth',1)
plot3(xh(vvv),ones(length(vvv),1)*xh(dd(2)),real(Vc(vvv,dd(2))),'k--','linewidth',1)
plot3(ones(length(vvv),1)*xh(dd(1)),xh(vvv),real(Vc(vvv,dd(1))),'k--','linewidth',1)
plot3(ones(length(vvv),1)*xh(dd(2)),xh(vvv),real(Vc(vvv,dd(2))),'k--','linewidth',1)
view(2)
axis tight

%
% Mark location of sensor with red red circle
%
vv=find(xh>xj,1,'first');
plot3(xh(vv),xh(vv),real(Vc(vv,vv)),'ro','Markersize',4,'MarkerFaceColor',[1 0 0]);
setlabel('x','x')


