close all;clear; set(0,'DefaultLineLineWidth',1,'DefaultLineMarkerSize',7);

%
%---------------------------------------------------------------------------------------
%
% Frequency response of reduced-models
%
% (a) Comparison of the frequency response of the full Ginzburg-Landau
% equation with three reduced-order models. Blue dashed lines represent the
% full model of order $n=220.$ The performance of reduced-order models based
% on $r=2,4$ and $6$ modes are shown in the (a), (b) and (c),
% respectively. Red lines represent balanced modes, black lines POD modes
% and green lines global eigenmodes. We observe that the balanced modes
% capture the peak value of the frequency response which represents the main
% characteristic of the input-output behavior. The approximation of the
% frequency response for the open-loop case is unsatisfactory for POD models
% of order $2$ and $4$ and for all global-mode models.
%
% (b) The frequency response of the closed feedback-loop based on a
% LQG-compensator. Blue dashed lines represent the full model of order
% $n=220.$ The performance of reduced-order models based on $r=2,4$ and $6$
% modes are shown in the (a), (b) and (c), respectively. Red lines represent
% balanced modes, black lines POD modes and green lines global
% eigenmodes. We observe that reduced order controller based on balanced
% modes outperforms the other two models. The poor performance of the
% reduced-order based on POD and global modes, is directly associated with
% the unsatisfactory approximation of the open-loop case in
% Figure~\ref{modred_1}.
%
%  Shervin Bagheri, 16 March 2008
% KTH Mechanics
% shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------
%

%
%  Parameters of the subcritical Ginzburg-Landau equation
%
nx = 220; 
[xh,A,Q,cd,U,mu0,mu2,cc,gam] = setup_subcritical(nx);

%
% Unstable domain: branch I and II
%
x1=-sqrt(-2*mu0/mu2);
x2=sqrt(-2*mu0/mu2);

%
% Input B1 (disturbances) at x=-11
% Covariance W=1
%
xw = -11; 
var = 0.4;
W = 1.;				
B1 = exp(-((xh-xw)/var).^2);

%
% Input B2 (actuator) at x=-3
% Control penalty R=1
%
xu = -3;
B2 = exp(-((xh-xu)/var).^2);
R = 1.;

%
% Output C1 (sensor) at x=-3
% Mes. noise G=1
%
w = diag(Q);
xs = 0;
C1 = exp(-((xh-xs)/var).^2).*w;C1 = C1';
G = 1.;

%
% Output C2 (objective) is energy (C2'*C2=Q)
%
C2 = sqrt(Q);

%
% Input and outputs
%
B = [B1 B2];
C = [C1; C2]; m=size(C,1);

%
% LQG design of full model
%
%
% Full-state information
%
Pc = care(A,sqrt(inv(R))*B2,Q);
K = -inv(R)*B2'*Pc;
Ac = A+B2*K;

%
% Estimation
%
Pe = care(A',C1'*sqrt(inv(G)),B1*W*B1');
L = -Pe*C1'*inv(G);

%
% Compensator
%
AA = [A B2*K; -L*C1 A+L*C1+B2*K];
BB = [B1 zeros(nx,1);zeros(nx,1) -L];
CC = [C2 zeros(m-1,nx) ; zeros(1,nx) R*K];

%
% Direct and adjoint global modes
%
Aa=Q\A'*Q;
[Us, ll] = oeig(A,Q);
[Ua, lla] = oeig(Aa,Q);
Ua2=Ua./(ones(nx,1)*diag(Ua'*Q*Us)');

%
% POD modes
%
V  =  lyap(A,B*B');
[POD,SPOD]=oeig(V);


%
% Balanced modes
%
Va = lyap(A',C'*C);
X = sqrth(V); X = X';
Y = sqrth(Va); Y = Y';
[UU,Sig,VV] =  svd(Y'*X);

%
% Construct reduced models of order 2,4 and 6
%
nred =[2 4 6];


fig1=figure('Units','Centimeters','Position',[1 1 8 15]);
fig2=figure('Units','Centimeters','Position',[1 1 8 15]);

for testind = 1:3

  r = nred(testind);

  %
  % Project system on POD modes
  %
  P1 = POD(:,1:r);
  E1 = SPOD(1:r);
  Ap = P1'*A*P1; 
  B2p = P1'*B2;
  B1p = P1'*B1;
  C1p = C1*P1; 
  C2p = C2*P1; 
  Qp= P1'*Q*P1;
  
  %
  % Full-state information (POD)
  %
  Ppc=care(Ap,sqrt(inv(R))*B2p,Qp);
  Kp=-inv(R)*B2p'*Ppc;
  Apc = Ap+B2p*Kp;

    
  %
  % Esimation (POD)
  %
  Ppe=care(Ap',C1p'*sqrt(inv(G)),B1p*W*B1p');
  Lp=-Ppe*C1p'*inv(G);
  ApL = Ap + Lp*C1p;
  
    
  %
  % Reduced-order controller (POD)
  %
  AAp=[A B2*Kp; -Lp*C1 Ap+Lp*C1p+B2p*Kp];
  BBp=[B1 zeros(nx,1);zeros(r,1) -Lp];
  CCp = [C2 zeros(m-1,r) ; zeros(1,nx) R*Kp];

  %
  % Project system on balanced modes
  %
  U1 = UU(:,1:r);
  V1 = VV(:,1:r);
  Sig1 = (Sig(1:r,1:r));
  T1 = X*V1*Sig1^(-0.5); 
  S1 = Y*U1*Sig1^(-0.5); 
  Ar = S1'*A*T1;
  B2r = S1'*B2;
  B1r = S1'*B1;
  C1r = C1*T1; 
  C2r = C2*T1; 
  Qr= T1'*Q*T1;

  %
  % Full-state information (balanced modes)
  %
  Prc=care(Ar,sqrt(inv(R))*B2r,Qr);
  Kr=-inv(R)*B2r'*Prc;
  Arc = Ar+B2r*Kr;

  %
  % Estimation (balanced modes)
  %
  Pre=care(Ar',C1r'*sqrt(inv(G)),B1r*W*B1r');
  Lr=-Pre*C1r'*inv(G);
  ArL = Ar + Lr*C1r;
  
  %
  % Reduced-order controller (balanced modes)
  %
  AAr=[A B2*Kr; -Lr*C1 Ar+Lr*C1r+B2r*Kr];
  BBr=[B1 zeros(nx,1);zeros(r,1) -Lr];
  CCr = [C2 zeros(m-1,r) ; zeros(1,nx) R*Kr];

  %
  % Project system on global modes
  %
  Usred = Us(:,1:r); % truncation
  Uared = Ua2(:,1:r); % truncation
  Am = diag(ll(1:r));
  B1m = Ua(:,1:r)'*Q*B1;
  B2m = Ua(:,1:r)'*Q*B2;
  C1m = C1*Usred; 
  C2m = C2*Usred; 
  Qm = Usred'*Q*Usred;

  %
  % Full-state information (global modes)
  %
  Pmc=care(Am,sqrt(inv(R))*B2m,Qm);
  Km=-inv(R)*B2m'*Pmc;
  Amc = Am+B2m*Km;
  
  %
  % Estimation (global modes)
  %
  Pme=care(Am',C1m'*sqrt(inv(G)),B1m*W*B1m');
  Lm=-Pme*C1m'*inv(G);
  AmL = Am + Lm*C1m;
  
  %
  % Reduced-order controller (global modes)
  %
  AAm=[A B2*Km; -Lm*C1 Am+Lm*C1m+B2m*Km];
  BBm=[B1 zeros(nx,1);zeros(r,1) -Lm];
  CCm = [C2 zeros(m-1,r) ; zeros(1,nx) R*Km];

  %
  % Frequency response
  %
  no=200;
  lax=linspace(-3,3,no);

  %
  % Open loop (no control)
  %
  RRo=zeros(no,1);
  RRor=zeros(no,1);
  RRop=zeros(no,1);
  RRom=zeros(no,1);
    
  %
  % Closed loop (with control)
  %
  RRc=zeros(no,1);
  RRcr=zeros(no,1);
  RRcp=zeros(no,1);
  RRcm=zeros(no,1);

for ind=1:no
  la=i*lax(ind);

  %
  % Open loop
  %
  RRo(ind)=max(svd(C2*inv(la*eye(nx)-A)*B1));
  RRor(ind)=max(svd(C2r*inv(la*eye(r)-Ar)*B1r));
  RRop(ind)=max(svd(C2p*inv(la*eye(r)-Ap)*B1p));
  RRom(ind)=max(svd(C2m*inv(la*eye(r)-Am)*B1m));
  
  %
  % Closed loop
  %
  RRc(ind) = max(svd(CC*inv(la*eye(2*nx)-AA)*BB));
  RRcr(ind) = max(svd(CCr*inv(la*eye(nx+r)-AAr)*BBr));
  RRcp(ind) = max(svd(CCp*inv(la*eye(nx+r)-AAp)*BBp));
  RRcm(ind) = max(svd(CCm*inv(la*eye(nx+r)-AAm)*BBm));

end

%
% Plot frequency response (uncontrolled system)
%

figure(fig1)
subplot(3,1,testind)
semilogy(lax,RRc,'b--',lax,RRcr,'r',lax,RRcp,'k',lax,RRcm,'g',lax,ones(length(lax),1),'k:')
xlim([-2 2]) 
setlabel('','|G_c|')
drawnow;

%
% Plot frequency response (controlled system)
%
figure(fig2)
subplot(3,1,testind)
semilogy(lax,RRo,'b--',lax,RRor,'r',lax,RRop,'k',lax,RRom,'g',lax,ones(length(lax),1),'k:')
xlim([-2 2]);
setlabel('','|G|')
drawnow;

end
