close all; clear; set(0,'DefaultLineLineWidth',1,'DefaultLineMarkerSize',5);
%
%---------------------------------------------------------------------------------------
%
%  Compare HSV of exact and snapshot-based balanced truncation
%
%  Hankel singular values of the approximate balanced truncation are marked with
%  crosses, the exact balanced truncation with black circles. The number of
%  singular values that are correctly captured increases with the number of
%  snapshots (red: 1000, green: 500 and blue: 70 snapshots).
%
%  Shervin Bagheri, 16 March 2008
%  KTH Mechanics
%  shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------
%

%
%  Parameters of the subcritical Ginzburg-Landau equation
%
nx = 220; 
[xh,A,Q,cd,U,mu0,mu2,cc,gam] = setup_subcritical(nx);

%
% Unstable domain: branch I and II
%
x1 = -sqrt(-2*mu0/mu2);
x2 = sqrt(-2*mu0/mu2);

% 
% Input B at branch I
%
var = 0.4;
B = exp(-((xh-x1)/var).^2);

%
% Output C at branch II
%
w = diag(Q);
C = exp(-((xh-x2)/var).^2).*w;C = C';  

%
% Adjoint state-space matrices
%
Aa = Q\A'*Q;
Ba = B'*Q;
Ca = Q\C';

%
% Exact Gramians
%
V  =  lyap(A,Aa,B*Ba);
Va = lyap(Aa,A,Ca*C);

%
% Balanced modes
%
r = 100;
X = sqrth(V);   X = X';
Y = sqrth(Va); Y = Y';
[UU,Sig,VV] =  svd(Y'*X);
U1 = UU(:,1:r);
V1 = VV(:,1:r);
Sig1 = (Sig(1:r,1:r));
T1 = X*V1*Sig1^(-0.5); 
S1 = Y*U1*Sig1^(-0.5); 

%
% Check if Gramians equal and diagonal
%
ss = 1:10;
Vhat = S1'*V*S1;
Vahat = T1'*Va*T1;

%
% Snapshot-based balanced truncation:
%  (i) 1000 snapshots (red)
%  (ii) 500 snapshots (green)
%  (iii) 70 snapshots (blue)
%
fac = [0.7, 1.4,  7];
col = ['r', 'g', 'b'];

mmm=40;
HSV=zeros(3,mmm);
for testind=1:3
  
  %
  % Time vector (the longer the better)
  %
  tmax = 1000;
  dt = 0.125*fac(testind);
  t = 0:dt:tmax;   nt = length(t);
  
  %
  % Compute impulse response of
  % direct and adjoint equations 
  %
  q = impulse(A,B,t);
  qa = impulse(A',Ca,t);

  % 
  % Take more snapshot when energy is large
  % and fewer as for long times
  %
  tind1=find(t>10,1);
  tind2=find(t>50,1);
  tind3=find(t>150,1);
  tind4=find(t>300,1);
  tindsnap=([1:4:tind1 tind1+1:1:tind2 tind2+1:4:tind3 tind3+1:24:tind4 tind4+1:48:nt]);
  ts = t(tindsnap);

  %
  % Time quadrature weights
  %
  wts = ([diff(ts) 0]+[0 diff(ts)])/2; 
  Qts = diag(wts);

  %
  % Snapshot vectors
  %
  X = q(:,tindsnap)*sqrt(Qts);
  Y = qa(:,tindsnap)*sqrt(Qts);
 
  %  
  % Snapshot based Balanced modes
  %
  [UUs,Sigs,VVs] =  svd(Y'*Q*X);
  T1s = X*VVs*Sigs^(-0.5);
  S1s = Y*UUs*Sigs^(-0.5);

  %
  % Check if Gramian equal and Diagonal
  %
  Vhat2 = S1s'*Q*(X*(X'*Q))*S1s;
  Vahat2= T1s'*Q*(Y*(Y'*Q))*T1s;

  %
  % Galerkin project: are the gramians equal & diagonal?
  %
  disp('Exact Case')
  diag(Vhat(ss,ss))
  diag(Vahat(ss,ss))
  diag(Sig1(ss,ss))
  
  disp('Snapshot based')
  diag(Vhat2(ss,ss))
  diag(Vahat2(ss,ss))
  diag(Sigs(ss,ss))
  
  HSV(testind,:) = diag(Sigs(1:mmm,1:mmm));
end

%
% Plot and compare exact HSV with snapshot-method
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 6]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);
semilogy(1:mmm,HSV(1,:),'ro',1:mmm,HSV(2,:),'go',1:mmm,HSV(3,:),'bo','MarkerSize',2)
hold on;  plot(1:mmm,diag(Sig(1:mmm,1:mmm)),'ko','MarkerSize',2)
grid off;
setlabel('n','HSV')
ylim([1e-15 1e3])


