close all;clear; set(0,'DefaultLineLineWidth',1,'DefaultLineMarkerSize',7);

%
%---------------------------------------------------------------------------------------
%
%  (a) The mean of the error covariance $P$ (lower dashed line) obtained by
%  solving the Riccati-equation \refeq{estimationriccati} is compared to the
%  estimation error (blue line) obtained by marching the estimator in time
%  \refeq{estimator}. Also, the mean value of the state (top dashed line/red
%  line) is shown and found to be at least three orders of magnitude larger
%  than the estimation error. It is evident that both the state and the
%  estimation error reach a steady state. 
%
%  (b) The rms-value of the error and the state are shown in
%  solid and dashed black lines, respectively. The red and green Gaussian
%  functions represent the location of the input (stochastic disturbances)
%  and the sensor. The error attains its minimum value just downstream of
%  the sensor location and increases upstream as well as downstream of
%  it. 
%
% Shervin Bagheri, 16 March 2008
% KTH Mechanics
% shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------

%
%  Parameters of the subcritical Ginzburg-Landau equation
%
nx = 220;
[xh,A,Q,cd,U,mu0,mu2,cc,gam] = setup_subcritical(nx);

%
% Unstable domain: branch I and II
%
x1 = -sqrt(-2*mu0/mu2);
x2 = sqrt(-2*mu0/mu2);

%
% Input B1 at x=-11
% Covariance W=1
%
var = 0.4;
xw = -11; 
B1 = exp(-((xh-xw)/var).^2);

%
% Output (Sensor) at x=0
% Measurement noise, alpha = 1
%
w = diag(Q);
xs = 0;
C2 = exp(-((xh-xs)/var).^2).*w;C2 = C2';
alpha=.1;                % mes. noise penalty

%
% Compuate error covariance and estimation gain
%
Y = care(A',C2',B1*B1',alpha^2);
L=-Y*C2'*inv(alpha^2);
Ae=A+L*C2;
mean_e = real(trace(Y*Q));
urms_e = sqrt(real(diag(Y)));

%
% Compute mean and rms of state
%
V = lyap(A,A',B1*B1'); 
mean = real(trace(V*Q));
urms = sqrt(real(diag(V)));

%
% March the estimation in time (Cranck Nicholson)
%

%
% Time vector
%
tmax = 500;
dt=0.5;tvec=0:dt:tmax;

%
% Initialize state vectors:
%  (i) estimated state (qe)
%  (ii) actual state (q)
%  (iii) error state (qh)
%
qe=zeros(nx,length(tvec));
q=zeros(nx,length(tvec));
qh=zeros(nx,length(tvec));


I=eye(nx);
Mmes=(I-dt/2*Ae);invMmes=inv(Mmes);
Mpes=(I+dt/2*Ae);

Mm=(I-dt/2*A);invMm=inv(Mm);
Mp=(I+dt/2*A);

%
% White noise for input (B1)
%
wh = randn(2,length(tvec))/sqrt(dt);

%
% Sensor noise
%
wh(2,:) = alpha*wh(2,:);

for tind=1:length(tvec)-1;
  
  %
  % State
  %
  E(tind) = real(diag(q(:,tind)'*Q*q(:,tind)));
  q(:,tind+1)=invMm*(Mp*q(:,tind)+dt/2*B1*(wh(1,tind+1)+wh(1,tind)));

  %
  % Estimate of State
  %
  Eh(tind) = real(diag(qh(:,tind)'*Q*qh(:,tind)));
  f=-L*C2*(q(:,tind+1)+q(:,tind))-L*(wh(2,tind+1)+wh(2,tind));
  qh(:,tind+1)=invMmes*(Mpes*qh(:,tind)+dt/2*f);
  
  %
  % Estimation Error
  %
  Ee(tind) = real(diag(qe(:,tind)'*Q*qe(:,tind)));
  qe(:,tind+1)=invMmes*(Mpes*qe(:,tind)+dt/2*B1*(wh(1,tind+1)+wh(1,tind))+dt/2*L*(wh(2,tind+1)+wh(2,tind)));
  
end
E(tind+1)=real(diag(q(:,tind+1)'*Q*q(:,tind+1)));
Eh(tind+1)=real(diag(qh(:,tind+1)'*Q*qh(:,tind+1)));
Ee(tind+1)=real(diag(qe(:,tind+1)'*Q*qe(:,tind+1)));

%
% Plot rms of estimation error and state vector
%

%
% Display B1 and C1 in figure
% with high resolution;
%
xx = linspace(-20,20,2000);
B1t = exp(-((xx-xw)/var).^2);
C2t = exp(-((xx-xs)/var).^2);

set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 12]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);

%
% Plot the mean of the error covariance (Fig 25)
%

subplot(2,1,1)
semilogy(tvec,real(mean)*ones(length(tvec),1),'k--',tvec,real(mean_e)*ones(length(tvec),1),'k--','LineWidth',1)
hold on;
plot(tvec,E,'r',tvec,Ee,'b'); hold off;
xlim([0 500])
setlabel('t','E')

mar=0.1;
subplot(2,1,2)
col1=0.9;
ymax=5.7;
ymin=0.0005;
fill([x1 x2 x2 x1 x1],[ymin ymin ymax-mar ymax-mar ymin],[col1 col1 col1],'Edgecolor',[col1 col1 col1]);
hold on;
colu=[0.2 0.8 0.2];
coly=[0.8 0.2 0.2];
fill(xx,B1t,colu,'edgecolor','none');
fill(xx,C2t,coly,'edgecolor','none');
plot(xh,abs(urms),'r',xh,real(urms_e),'b')
axis([-20 20 0 ymax])
setlabel('x','');



