close all; clear; set(0,'DefaultLineLineWidth',1,'DefaultLineMarkerSize',4);

%
%---------------------------------------------------------------------------------------
%
% The controlled Ginzburg-Landau equation with stochastic excitation: 
%
% (a) white noise $w$ with zero mean and unit variance $W=1$ forces the
% system at $x=-11$, just upstream of unstable region with input $B_1$ as a
% Gaussian function (green). Measurements $y(t)$ of the state (red Gaussian)
% contaminated by white noise with zero mean and variance $G=0.1$ are taken
% at $x_s=0.$ The actuator $u$ with control penalty $R=1$ is placed upstream
% of the sensor at $x_u=-3.$ The rms-values of the uncontrolled and
% LQG-controlled state are given by the solid red and black lines,
% respectively.
%
%  The absolute value of the state $|q|$ is shown in an $x$-$t$-plane in
% (b), while the lower plot (c) displays the kinetic energy $E=\|q\|_M$ as a
% function of time. The control is only engaged for $t\in [250,750].$ Dashed
% lines in (c) indicate the mean value computed from Lyapunov equation.
%
% Shervin Bagheri, 16 March 2008
% KTH Mechanics
% shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------


%
%  Parameters of the subcritical Ginzburg-Landau equation
%
nx = 220;
[xh,A,Q,cd,U,mu0,mu2,cc,gam,muc] = setup_subcritical(nx);

%
% Unstable domain: branch I and II
%
x1=-sqrt(-2*(mu0-cc^2)/mu2);
x2=sqrt(-2*(mu0-cc^2)/mu2);

%
% Input B1 (disturbances) at x=-11
% Covariance W=1
%
var = 0.4;
xw = -11;
B1 = exp(-((xh-xw)/var).^2);

%
% Input B2 (actuator) at x=-3
% Control penalty l=1
% Control weight  R=1
%
xu = -3;
B2 = exp(-((xh-xu)/var).^2);
R  = 1;
l=1.;			


%
% Output C1 (objective) is energy (C2'*C2=Q)
%
C1 = sqrth(Q); 

%
% Output C2 (sensor) at x=-3
% Mes. noise alpha=1
%
xs = 0;
C2 = exp(-((xh-xs)/var).^2).*diag(Q); C2 = C2';
alpha=.1;         

%
% Standard formulation (See book of zhou)
%
G  = [1 0;0 alpha];               
D12 = l*sqrt(R);
D21 = [0 1]*G;
B1 = [exp(-((xh-xw)/var).^2) zeros(nx,1)]*G;
B = [B1  B2]; r = size(B,2);
C = [C1; C2]; m = size(C,1);
D = [zeros(m-1,r-1) D12*ones(m-1,1); D21 0 ];

%
% Full-state information 
%
X2 = care(A,B2,C1'*C1,l^2);
K2 = -inv(l^2)*inv(R)*B2'*X2;

%
% Estimation 
%
Y2 = care(A',C2',B1*B1',alpha^2);
L2 = -Y2*C2'*inv(alpha^2);

%
% Closed-loop (compensator)
%
AA = A+L2*C2+B2*K2;
AA2 = [A B2*K2; -L2*C2 AA];
BB2 = [B1; zeros(nx,1) -L2];
CC2 = [C1 zeros(m-1,nx) ; zeros(1,nx) D12*K2];

%
% Statistics (Covariance, rms and mean) of 
% uncontrolled state
%
V  =  mylyap(A,A',B1*B1');
rms = sqrt(real(diag(V)));
mean=real(trace(V*Q));

%
% Statistics (Covariance, rms and mean) of 
% controlled state
%
VV_H2 = mylyap(AA2,AA2',BB2*[1 0;0 0]*BB2');
rms_H2 = sqrt(real(diag(VV_H2(1:nx,1:nx))));
mean_H2=real(trace(VV_H2(1:nx,1:nx)*Q));

%%
%% March the system in time (Cranck Nicholson)
%%

%
% Time vector
%
tmax = 1000;
dt = 0.1; t = 0:dt:tmax; 
nt = length(t);

%
% Initialize state vector
%
q2   = zeros(2*nx,nt);

%
% Cranck-Nicholson Matrices
%
I = eye(nx);
Mm = (I-dt/2*A);
invMm = inv(Mm);
Mp = (I+dt/2*A);

II = eye(2*nx);
Mm2 = (II-dt/2*AA2);
invMm2 = inv(Mm2);
Mp2 = (II+dt/2*AA2);

%
% White noise
%
wh = randn(1,length(t))/sqrt(dt);

%
% Turn control on between [250,750]
%
tind_on = find(t>=250,1,'first');
tind_off = find(t>=750,1,'first');
vec = 1:nx;

for tind=1:length(t)-1
  E(tind) = real(diag(q2(vec,tind)'*Q*q2(vec,tind)));

  if tind>=tind_on && tind<=tind_off 
    if tind==tind_on; disp('control on'); end
    q2(:,tind+1) = invMm2*(Mp2*q2(:,tind)+dt/2*BB2*([wh(tind+1)+wh(tind); 0]));
  else
    if tind==tind_off; disp('control off'); end
    q2(vec,tind+1) = invMm*(Mp*q2(vec,tind)+dt/2*B1(:,1)*(wh(tind+1)+wh(tind)));
  end

end


%
% Plot rms of controlled and uncontrolled state
%

%
% Display B1,B2 and C1 in figure
% with high resolution;
%
xx = linspace(-20,20,2000);
B1t = exp(-((xx-xw)/var).^2);
B2t = exp(-((xx-xu)/var).^2);
C2t = exp(-((xx-xs)/var).^2);

set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 12]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);

subplot(10,1,1:5)
ymax=7;
fill([x1 x2 x2 x1 x1], [-ymax -ymax ymax ymax -ymax],[0.9 0.9 0.9],'edgecolor',[0.9 0.9 0.9])
hold on;
colu=[0.2 0.8 0.2];
coly=[0.8 0.2 0.2];
fill(xx,B1t,colu,'edgecolor','none');
fill(xx,B2t,colu,'edgecolor','none');
fill(xx,C2t,coly,'edgecolor','none');
plot(xh,abs(rms),'r',xh,abs(rms_H2),'k','LineWidth',1)
xlim([-15 20]); grid off
ylim([0 ymax]);
setlabel('x','rms')
%
% Plot textboxes
%
annotation1 = annotation(  fig1,'textbox',  'Position',[0.195 0.1858 0.055 0.11],  'EdgeColor','none',  'FitHeightToText','off',  'FontSize',7,  'String',{'w'}); 
annotation2 = annotation(  fig1,'textbox',  'Position',[0.375 0.1858 0.055 0.11],  'EdgeColor','none',  'FitHeightToText','off',  'FontSize',7,  'String',{'u'});
annotation3 = annotation(  fig1,'textbox',  'Position',[0.4733 0.1725 0.055 0.11],  'EdgeColor','none',  'FitHeightToText','off',  'FontSize',7,  'String',{'y'});

%
% Plot spatio-temporal evolution of state
%
subplot(10,1,6:8)
MM = 60;
surf(t,xh(MM:nx-MM),abs(q2(MM:nx-MM,:)));
ccolor; view(2); setlabel('','x'); axis tight

%
% Plot evolution of energy 
%
subplot(10,1,9:10)
fill([250 250 750 750], [3e-2 5e3 5e3 3e-2],[0.9 0.9 0.9],'EdgeColor',[0.9 0.9 0.9])
hold on;
semilogy(t(1:end-1),E,'k');
plot(t,ones(length(t),1)*mean,'r--',t,ones(length(t),1)*mean_H2,'g--')
ylim([3e-2 5e3]);hold off; setlabel('t','E')

