close all; clear; set(0,'DefaultLineLineWidth',1,'DefaultLineMarkerSize',4);

%
%---------------------------------------------------------------------------------------
%
%  Actuator and sensor placement for the supercritical
%  Ginzburg-Landau equation which yields a stabilizable and
%  detectable system. The spatial support of the actuator (blue
%  bar), sensor (red bar), the unstable domain (gray region) as
%  well as the unstable global mode (black lines) together with its
%  corresponding adjoint mode (red lines) are shown
%  
%  The spatio-temporal response to an impulse in time
%  induced at $x=-10$ for the uncontrolled system (a) and
%  LQG-controlled system (c)
%  
%  Top figure: The perturbation energy of an initial
%  condition which illustrates the asymptotic growth and decay of
%  the global mode of the controlled and uncontrolled systems.
%  Bottom figure: The spectrum of the uncontrolled (red circles)
%  and LQG-controlled (black crosses) Ginzburg-Landau equation.
%  The exponential growth of the wavepacket in Figure~\ref{super2
%  a}a is due to one unstable global mode of the open-loop shown by
%  the red circle in the unstable half-plane (gray region). The
%  LQG-based closed-loop is stable with no unstable eigenvalues.
%
% Shervin Bagheri, 16 March 2008
% KTH Mechanics
% shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------
%

%
%  Parameters of the supercritical Ginzburg-Landau equation
%
nx  = 220;
[xh,A,Q,cd,U,mu0,mu2,cc,gam,muc] = setup_supercritical(nx);

%
% Unstable domain: branch I and II
%
x1=-sqrt(-2*mu0/mu2);
x2=sqrt(-2*mu0/mu2);

%
% Direct and adjoint global modes
%
Aa =Q\A'*Q;
[Us, ll] = oeig(A,Q);
[Ua, lla] = oeig(Aa,Q);
Ua2=Ua./(ones(nx,1)*diag(Ua'*Q*Us)');

%
% Input B1 (disturbances) at x=-11
% Covariance W=1
%
var = 0.1;
xw = -11;
B1 = exp(-((xh-xw)/var).^2);

%
% Input B2 (actuator) at branch I
% Control penalty l=1
% Control weight  R=1
%
xu = x1;
B2 = exp(-((xh-xu)/var).^2);
R  = 1;                       
l=1;			

%
% Output C1 (objective) is energy (C1'*C1=Q)
%
C1 = sqrth(Q); 

%
% Output C2 (sensor) at x=-3
% Mes. noise alpha=1
%
xs = x2;
C2 = exp(-((xh-xs)/var).^2).*diag(Q); C2 = C2';
alpha=1.;                

%
% Standard formulation (See book of zhou)
%
G  = [1 0;0 alpha];               
D12 = l*sqrt(R);
D21 = [0 1]*G;
B1 = [exp(-((xh-xw)/var).^2) zeros(nx,1)]*G;
B = [B1 B2]; r = size(B,2);
C = [C1; C2]; m = size(C,1);
D = [zeros(m-1,r-1) D12*ones(m-1,1); D21 0 ];

%
% Full-state information 
%
X2 = care(A,B2,C1'*C1,l^2);
K2 = -inv(l^2)*inv(R)*B2'*X2;

%
% Estimation 
%
Y2 = care(A',C2',B1*B1',alpha^2);
L2 = -Y2*C2'*inv(alpha^2);

%
% Closed-loop (compensator)
%
AA2 = [A B2*K2; -L2*C2 A+L2*C2+B2*K2];
BB2 = [B1; zeros(nx,1) -L2];
CC2 = [C1 zeros(m-1,nx) ; zeros(1,nx) D12*K2];

%
% eigenvalues of closed loop
%
QQ=[Q zeros(nx,nx); zeros(nx,nx) Q];
[UUs, ll2] = oeig(AA2,QQ);

%%
%% March the system in time (Exponential matrix)
%%

%
% Time vector
%
tmax = 100;
dt = 0.1; t = 0:dt:tmax; 
nt = length(t);


%
% Initialize state vectors
%  (i) uncontrolled (q)
%  (ii) controlled  (q1)
%
q   = zeros(nx,nt);
q2   = zeros(2*nx,nt);


expmat = expm(A*dt);
H = eye(nx);
expmat2 = expm(AA2*dt);
H2 = eye(2*nx);

for tind=1:length(t)
  
  H = H*expmat;
  q(:,tind) = H*B1(:,1);
  E(tind) = real(diag(q(:,tind)'*Q*q(:,tind)));

  H2 = H2*expmat2;
  q2(:,tind) = H2*[B1(:,1);zeros(nx,1)];
  E2(tind) = real(diag(q2(1:nx,tind)'*Q*q2(1:nx,tind)));

end

%
% Plot of controllability modal residual (Fig 20)
%
%figure(2);
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 5]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);
hold on;
MM=20;
plot(1:MM,1./abs(diag(Ua(:,1:MM)'*Q*Us(:,1:MM))),'b-o','Markersize',5);
plot(1:MM,abs(Ua(:,1:MM)'*Q*B2),'r-o','Markersize',5);
plot(1:MM,abs(Ua2(:,1:MM)'*Q*B2),'k-o','Markersize',5);
setlabel('n','')
xlim([1 20])
box on;

%
% Plot of spatial support of actuator (sensor) with adjoint (global) modes
% (Fig 29)
%
ymax= 1.19;
L=25;
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 5]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);
MM=1;
colu=[0.1 0.1 0.8];
cols=[0.8 0.1 0.1];
fill([x1 x2 x2 x1 x1], [-ymax -ymax ymax ymax -ymax],[0.9 0.9 0.9],'edgecolor',[0.9 0.9 0.9])
hold on;
fill([xu-2*var xu+2*var xu+2*var xu-2*var xu-2*var],[ 0 0 ymax ymax 0],colu,'edgecolor',colu);
fill([xs-2*var xs+2*var xs+2*var xs-2*var xs-2*var],[ 0 0 ymax ymax 0],cols,'edgecolor',cols);
plot(xh,abs(Us(:,MM))/max(abs(Us(:,MM))),'k',xh,real(Us(:,MM))/max(abs(Us(:,MM))),'k--',xh,-abs(Us(:,MM))/max(abs(Us(:,MM))),'k');axis([-xh(end) xh(end) -1.2 1.2 ]);
plot(xh,abs(Ua(:,MM))/max(abs(Ua(:,MM))),'r',xh,real(Ua(:,MM))/max(abs(Ua(:,MM))),'r--',xh,-abs(Ua(:,MM))/max(abs(Ua(:,MM))),'r');axis([-xh(end) xh(end) -1.2 1.2 ]);
plot(xh,zeros(nx),'k')
xlim([-L L]);ylim([.0 ymax+0.02])
hold off;
setlabel('x','')

%
% Plot of spatio-temporal state evolution of open and closed -loop (Fig 30)
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 10]);
subplot(2,1,1)
MM = 60;
surf(t,xh(MM:nx-MM),real(q(MM:nx-MM,:)));
ccolor;
view(2);
setlabel('','x');
axis tight

subplot(2,1,2)
surf(t,xh(MM:nx-MM),real(q2(MM:nx-MM,:)));
ccolor;
view(2);
setlabel('t','x');
axis tight

%
% Plot of spectrum  and energy of open and closed-loop (Fig 31)
%
mar = 0.001;
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 12]);
subplot(2,1,1)
plot(t,E,'k',t,E2,'r','Linewidth',2)
axis([0 100 2e-4 2e2])
setlabel('t','E')

subplot(2,1,2)
fill([0 0 .2-5*mar .2-5*mar],[-1+mar 0.5-10*mar .5-10*mar -1+mar],[0.9 0.9 0.9],'EdgeColor',[0.9 0.9 0.9])
hold on
plot(real(ll),imag(ll),'ro',real(ll2),imag(ll2),'ko','Markersize',4);
axis([-2 0.2 -1 0.5])
hold off
setlabel('\lambda_r','\lambda_i')



