close all;clear; set(0,'DefaultLineLineWidth',1,'DefaultLineMarkerSize',7);

%
%---------------------------------------------------------------------------------------
%
% Model reduction error of the POD (black circles), balanced (red crosses)
% and global (green triangles) modes. For the balanced modes the error
% always decays by increasing the number of modes, in contrast to the error
% of POD modes. The error does not decay at all for the first 50 global
% modes due to the failure to project the input $B$ located upstream of
% branch I onto the global eigenmodes located close and downstream to branch
% II.
%
% Shervin Bagheri, 16 March 2008
% KTH Mechanics
% shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------
%


%
%  Parameters of the subcritical Ginzburg-Landau equation
%
nx = 220;  
[xh,A,Q,cd,U,mu0,mu2,cc,gam] = setup_subcritical(nx);

%
% Unstable domain: branch I and II
%
x1=-sqrt(-2*mu0/mu2);
x2=sqrt(-2*mu0/mu2);

% 
% Input B at x=-11
%
var = 0.4;
xw = -11; 
B1 = exp(-((xh-xw)/var).^2);

%
% Output C is energy, C'*C = Q
%
C2 = sqrt(Q);

%
% Forward and adjoint global modes
%
Aa=Q\A'*Q;
[Us, ll] = oeig(A,Q);
[Ua, lla] = oeig(Aa,Q);
Ua2=Ua./(ones(nx,1)*diag(Ua'*Q*Us)');

%
% POD modes
%
V  =  lyap(A,B1*B1');
[POD,SPOD]=oeig(V);

%
% Balanced truncation
%
Va = lyap(A',C2'*C2);
X = sqrth(V); X = X';
Y = sqrth(Va); Y = Y';
[UU,Sig,VV] =  svd(Y'*X);

%
% Compute model reduction error
%
nred = 1:1:50;
for testind = 1:length(nred)

  %% 1.  POD  modes 
  r = nred(testind);
  P1 = POD(:,1:r);
  Ap = P1'*A*P1; 
  B1p = P1'*B1;
  C2p = C2*P1; 

  
  %% 2. Balanced truncation
  U1 = UU(:,1:r);
  V1 = VV(:,1:r);
  Sig1 = (Sig(1:r,1:r));
  T1 = X*V1*Sig1^(-0.5);
  S1 = Y*U1*Sig1^(-0.5);
  Ar = S1'*A*T1;
  B1r = S1'*B1;
  C2r = C2*T1; 

  %%3. global modes
  Usred = Us(:,1:r);
  Uared = Ua2(:,1:r);
  Am = diag(ll(1:r));
  B1m = Ua(:,1:r)'*Q*B1;
  C2m = C2*Usred; 

  %% Harmonic response
  no=200;
  lax=linspace(-3,3,no);
  RRo=zeros(no,1);
  RRor=zeros(no,1);
  RRop=zeros(no,1);
  RRom=zeros(no,1);
  
  for ind=1:no
    la=i*lax(ind);
  
    RRo(ind)=max(svd(C2*inv(la*eye(nx)-A)*B1));
    RRor(ind)=max(svd(C2r*inv(la*eye(r)-Ar)*B1r));
    RRop(ind)=max(svd(C2p*inv(la*eye(r)-Ap)*B1p));
    RRom(ind)=max(svd(C2m*inv(la*eye(r)-Am)*B1m));
    
  end

  %% Infinity-error
  nr(testind)=max(abs(RRo-RRor));
  np(testind)=max(abs(RRo-RRop));
  nm(testind)=max(abs(RRo-RRom));

end

%
% Plot the error norm
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 7]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);
semilogy(nred(1:testind),nr(1:testind),'rx',nred(1:testind),np(1:testind),'ko','Markersize',2,'LineWidth',1);
hold on;semilogy(nred(1:testind),nm(1:testind),'gv','Markersize',2,'LineWidth',1);hold off
axis([0 50 1e-9 1e3])
grid off
setlabel('n','||G-G_r||_\infty')




