close all;clear; set(0,'DefaultLineLineWidth',1,'DefaultLineMarkerSize',4);

%
%---------------------------------------------------------------------------------------
%
% 
%  Comparison of the frequency response of the open-loop
%    (a), LQG-controlled (b) and $\mathcal{H}_\infty$-controlled (c)
%    Ginzburg-Landau equation. The $\infty$-norm corresponding to the
%    peak value of the response (dashed line) and the 2-norm
%    corresponding the to integral of the response (gray region) are
%    displayed in each figure.
%
% (a) Top figure: The perturbation energy of an initial
%   condition which illustrates the asymptotic growth and decay of
%   the global mode of the controlled and uncontrolled systems.
%   Bottom figure: The spectrum of the uncontrolled (red circles)
%   and LQG-controlled (black crosses) Ginzburg-Landau equation.
%   The exponential growth of the wavepacket in Figure~\ref{super2
%   a}a is due to one unstable global mode of the open-loop shown by
%   the red circle in the unstable half-plane (gray region). The
%   LQG-based closed-loop is stable with no unstable eigenvalues. }
%
% Shervin Bagheri, 16 March 2008
%
%---------------------------------------------------------------------------------------

%
%  Parameters of the subcritical Ginzburg-Landau equation
%
nx = 220;            % Resolution i x-direction
[xh,A,Q,cd,U,mu0,mu2,cc,gam,muc] = setup_subcritical(nx);

%
% Unstable domain: branch I and II
%
x1=-sqrt(-2*mu0/mu2);
x2=sqrt(-2*mu0/mu2);

%
% Input B1 (disturbances) at x=-11
% Covariance W=1
%
var = 0.4;
xw = -11;
B1 = exp(-((xh-xw)/var).^2);

%
% Input B2 (actuator) at branch I
% Control penalty l=1
% Control weight  R=1
%
xu = -3;
B2 = exp(-((xh-xu)/var).^2);
R  = 1;                       
l  = 1;			

%
% Output C2 (sensor) at x=-3
% Mes. noise alpha=1
%
xs = 0;
C2 = exp(-((xh-xs)/var).^2).*diag(Q); C2 = C2';
alpha=1.;


%
% Output C1 (objective) is energy (C1'*C1=Q)
%
C1 = sqrth(Q); 

%
% Standard formulation (See book of zhou)
%
G  = [1 0;0 alpha];
D12 = l*sqrt(R);
D21 = [0 1]*G;
B1 = [exp(-((xh-xw)/var).^2) zeros(nx,1)]*G;
B = [B1 B2]; r = size(B,2);
C = [C1; C2]; m = size(C,1);
D = [zeros(m-1,r-1) D12*ones(m-1,1); D21 0 ];


%
% Hinf Control
%
disp(' Hinf-Control');
%
% Find minimum gamma by Bisection Algorithm
%
gmax = 50; gmin = 0; git = 1;

git=10;
for ind=1:git
  gamma=(gmax+gmin)/2;

  %
  % Full-state control
  %
  RR  = [-gamma^2 0 0;       0 1 0;       0 0 l^2];
  [Xinf,tmp,tmp,status1] = care(A,B,C1'*C1,RR);

  %
  % Estimation
  %
  RR2 = [-gamma^2*eye(m-1) zeros(m-1,1) ; zeros(1,m-1) alpha^2];
  [Yinf,tmp,tmp,status2] = care(A',C',B1*B1',RR2);

  %
  % Find minimum gamma
  %
  if  (status1 ~= -1 && status2 ~=-1)
    lt1 = max(abs(eig(Xinf*Yinf)))-gamma^2; 
% $$$     lt2 = min(eig(Xinf));        % X>0? 
% $$$     lt3 = min(eig(Yinf));        % Y>0?
    lt2=1; lt3=1;
    if (lt1<=0 && lt2>=-eps && lt3>=-eps)
      disp(['     iteration: ' num2str(ind) '/' num2str(git) '  gamma:  ' num2str(gamma) '   - Solution Accepted'])
      gmax = gamma;
    else
      disp(['     iteration: ' num2str(ind) '/' num2str(git) '  gamma:  ' num2str(gamma) '   - Solution Rejected'])
      gmin = gamma;
    end
  else
    disp(['     iteration: ' num2str(ind) '/' num2str(git) '  gamma:  ' num2str(gamma) '   - Solution Rejected'])
    gmin = gamma;
  end
end


%
% Minimum gamma that yeilds a solution
%
gamma = gmax;
disp(['minimum gamma: ' num2str(gamma)])
%
% Full-state control (Hinf)
%
RR  = [-gamma^2 0 0;       0 1 0;       0 0 l^2];
Xinf = care(A,B,C1'*C1,RR);
Kinf = -inv(l^2)*inv(R)*B2'*Xinf;

%
% Estimation (Hinf)
%
RR2 = [-gamma^2*eye(m-1) zeros(m-1,1) ; zeros(1,m-1) alpha^2];
Yinf = care(A',C',B1*B1',RR2);
Zinf =  inv(eye(nx) -1/gamma^2*Yinf*Xinf);
Linf = -inv(alpha^2)*Zinf*Yinf*C2'; 

%
% Closed-loop (Hinf)
%
AAinf = [A B2*Kinf; -Linf*C2 A+Linf*C2+B2*Kinf+1/gamma^2*B1*B1'*Xinf];
BBinf = [B1 ;zeros(nx,1) -Linf];
CCinf = [C1 zeros(m-1,nx) ; zeros(1,nx) D12*Kinf];


[Og]=lyap(AAinf',CCinf'*CCinf); Bdash=BBinf'*Og*BBinf; 
Tinfnrm = abs(sqrt(trace(Bdash)));    

%
% LQG/H2 control
%

%
% Full-state control (H2)
%
X2 = care(A,B2,C1'*C1,l^2);
K2 = -inv(l^2)*inv(R)*B2'*X2;

%
% Estimation (H2)
%
Y2 = care(A',C2',B1*B1',alpha^2);
L2 = -Y2*C2'*inv(alpha^2);

%
% Closed-loop (H2)
%
AA2 = [A B2*K2; -L2*C2 A+L2*C2+B2*K2];
BB2 = [B1; zeros(nx,1) -L2];
CC2 = [C1 zeros(m-1,nx) ; zeros(1,nx) D12*K2];

[Og]=lyap(AA2',CC2'*CC2); Bdash=BB2'*Og*BB2; 
T2nrm = abs(sqrt(trace(Bdash)));    

%
% Compute frequency response of open, closed (H2) and closed (Hinf) loops
%
no = 100; nn = 2;
lax=linspace(-nn,nn,no);

for ind=1:no
    iw=i*lax(ind);
    
    sG(ind)= max(svd(C*inv(iw*eye(nx)-A)*B));
    sGc2(ind) = max(svd(CC2*inv(iw*eye(2*nx)-AA2)*BB2));
    sGcinf(ind) = max(svd(CCinf*inv(iw*eye(2*nx)-AAinf)*BBinf));
end

T = max(sG);
T2_inf_svd= max(sGc2);

%
% Compute optimal growth of open, closed (H2) and closed (Hinf) loops
%
[F,Finv] = sqrth(Q);
[F2,Finv2] = sqrth([Q zeros(nx,nx); zeros(nx,nx) Q]);

%
% Time vector
%
dt = 1; tvec = 0:dt:100;

%
% March in time (exponential matrix)
%
expmat = expm(A*dt);
H = eye(nx);

expmat2 = expm(AA2*dt);
H2 = eye(2*nx);

expmat3 = expm(AAinf*dt);
H3 = eye(2*nx);

for tind=1:length(tvec)
  
  H = H*expmat;
  [UU,SS,VV] = svd(F*H*Finv);
  Gr(tind) = SS(1,1)^2;
  qopt(:,tind) = VV(:,1);

  H2 = H2*expmat2;
  [UU2,SS2,VV2] = svd(F2*H2*Finv2);
  Gr2(tind) = SS2(1,1)^2;
  qopt2(:,tind) = VV2(:,1);

  H3 = H3*expmat3;
  [UU3,SS3,VV3] = svd(F2*H3*Finv2);
  Gr3(tind) = SS3(1,1)^2;
  qopt3(:,tind) = VV3(:,1);

end
[t0,ind]=max(Gr);
q0 = Finv*qopt(:,ind);


%
% Plot transfer function norm (fig27)
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 12]);
axes1 = axes('FontName','arial', 'FontSize',7, 'Parent',fig1);
subplot(2,1,1)
semilogy(lax,ones(no,1),'k--',lax,sG,'r',lax,sGc2,'k',lax,sGcinf,'b','LineWidth',1)
xlim([-nn nn])
setlabel('\omega','|G|')

%
% Plot optimal growth (fig28)
%
subplot(2,1,2)
set(0,'Units','Centimeters');
plot(tvec, Gr/Gr(1),'r',tvec,Gr2/Gr2(1),'k',tvec,Gr3/Gr3(1),'b','Linewidth',1)
axis([0 100 0 120])
setlabel('t','E')
