close all;clear; set(0,'DefaultLineLineWidth',1,'DefaultLineMarkerSize',7);
%
%---------------------------------------------------------------------------------------
%
%  Global stability concepts:
% 
%: (a) The spatio-temporal evolution of a disturbance in a 
%       globally unstable flow. The disturbance grows exponentially until the
%       cubic nonlinear term -|q|^2q of the Ginzburg-Landau equation causes the
%       disturbance to saturate and oscillate.
%
% (b) The energy that corresponds to the evolution in (a) is shown in red, 
%      and the linear exponential growth for the linear Ginzburg-Landau equation
%      is shown in dashed black.
%
%
%  Shervin Bagheri, 16 March 2008
%  KTH Mechanics
%  shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------
%


%
%  Parameters of the supercritical Ginzburg-Landau equation
%
nx = 220;            
[xh,A,M,cd,U,mu0,mu2,cc,gam] = setup_supercritical(nx);

%
% Time vector
%
tmax=200;  dt = 1;
tvec = 0:dt:tmax;
nt = length(tvec);

%
% Compute optimal disturbance 
%
[F,Finv] = sqrth(M);
expmat1 = expm(A*dt);
H = eye(nx);
for tind=1:nt
  H = H*expmat1;
  [U,S,V] = svd(F*H*Finv);
  G(tind) = S(1,1)^2;
  qopt(:,tind) = V(:,1);
end
[t0,ind] = max(G);
q0 = Finv*qopt(:,ind);

%
% Unstable domain: branch I and II
%
x1 = -sqrt(-2*mu0/mu2);
x2 = sqrt(-2*mu0/mu2);

%
% Initialize state vectors
%
q = zeros(nx,nt);
q(:,1) = q0;
qlin = zeros(nx,nt);
qlin(:,1) = q0;

%
% March in time (Cranck Nicholson scheme)
%
I=eye(nx);
Mm=(I-dt/2*A);invMm=inv(Mm);
Mp=(I+dt/2*A);

%
% Coefficient of nonlinear term
%
a = .01;

%
% White noise
%
wh=randn(1,nt)/sqrt(dt);
wh = zeros(1,nt);

for tind=1:nt-1;
  
  %
  % nonlinear term
  %
  f(:,tind) = -a*abs(q(:,tind)).^2.*q(:,tind);

  %
  % Step forward 
  %
  E(tind) = real(diag(q(:,tind)'*M*q(:,tind)));
  q(:,tind+1) = invMm*(Mp*q(:,tind)+dt*f(:,tind)+dt/2*q0*(wh(1,tind+1)+wh(1,tind)));

  Elin(tind) = real(diag(qlin(:,tind)'*M*qlin(:,tind)));
  qlin(:,tind+1) = invMm*(Mp*qlin(:,tind)+dt/2*q0*(wh(1,tind+1)+wh(1,tind)));

end
E(tind+1) = real(diag(q(:,tind+1)'*M*q(:,tind+1)));
Elin(tind+1) = real(diag(qlin(:,tind+1)'*M*qlin(:,tind+1)));

%
% Plot spatio-temporal evolution of disturbance (Fig 5)
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 10]);
axes1 = axes('FontName','arial', 'FontSize',6, 'Parent',fig1);
subplot(8,1,[1:6]);
MM=70;
surf(tvec,xh(MM:end-MM),real(q(MM:end-MM,:)));
ccolor; hold on;

%
% Find and plot branch I and II
%
xlim([0 tmax]);
dd(1)=find(xh>x1,1,'first');
dd(2)=find(xh>x2,1,'first');
vvv=dd(1):dd(2);
plot3(tvec,xh(dd(1))*ones(nt,1),abs(q(dd(1),:)),'k--','linewidth',1)
plot3(tvec,xh(dd(2))*ones(nt,1),abs(q(dd(2),:)),'k--','linewidth',1)
annotation1 = annotation(fig1,'textbox', 'LineStyle','none','Position',[0.15 0.28 0.5 0.5], 'FontName','Arial',  'FontSize',7,  'String',{'(a)'});
view([-0.5 -0.6 8]);  axis tight; grid off;  zlim([0 5]);  
%set(gca, 'ytick',[-20 -10 0 10 20])
setlabel('t','x')

%
% Plot energy evolution in time
%
subplot(8,1,[7:8]); 
semilogy(tvec,Elin,'k--',tvec(1:tind),E(1:tind),'r');
annotation1 = annotation(fig1,'textbox', 'LineStyle','none','Position',[0.15 -0.2 0.5 0.5], 'FontName','Arial',  'FontSize',7,  'String',{'(b)'});
axis([0 tmax 0 1e5]);  
setlabel('t','E')
set(gca, 'ytick',[0 10 1000])




