close all; clear; set(0,'DefaultLineLineWidth',1,'DefaultLineMarkerSize',7);
%
%---------------------------------------------------------------------------------------
%
%  Global stability concepts:
% 
% (c) Linear transient growth of a perturbation in space and time: 
%      an optimal initial perturbation grows as it enters the unstable domain at
%      branch I at x=-8.2 until it reaches branch II at x=8.2. The two dashed
%      lines depict branch I and II.
%
% (d) The corresponding optimal energy growth of the convectively unstable flow
%       in (c).
%
%
%  Shervin Bagheri, 16 March 2008
%  KTH Mechanics
%  shervin@mech.kth.se
%
%---------------------------------------------------------------------------------------
%


%
%  Parameters of the subcritical Ginzburg-Landau equation
%
nx = 220;
[xh,A,M,cd,U,mu0,mu2,cc,gam] = setup_subcritical(nx);

%
% Time vector
%
tmax=100;
dt = 1;
tvec = 0:dt:tmax;

%
% Compute optimal disturbance 
%
[F,Finv] = sqrth(M);
expmat1 = expm(A*dt);
H = eye(nx);
for tind=1:length(tvec)
  H = H*expmat1;
  [U,S,V] = svd(F*H*Finv);
  G(tind) = S(1,1)^2;
  qopt(:,tind) = V(:,1);
end
[t0,ind] = max(G);
q0 = Finv*qopt(:,ind);

%
% Unstable domain: branch I and II
%
x1 = -sqrt(-2*(mu0-cc^2)/mu2);
x2 = sqrt(-2*(mu0-cc^2)/mu2);

%
% Increase time resolution
%
tmax = 300;  dt =.1;
t = 0:dt:tmax;  nt = length(t);

%
% Initialize state vectors
%
q   = zeros(nx,nt);
q(:,1) = q0;

%
% March in time (Cranck Nicholson scheme)
%
I = eye(nx);
Mm = (I-dt/2*A);  invMm = inv(Mm);
Mp = (I+dt/2*A);
for tind=1:length(t)-1
  E(tind) = real(diag(q(:,tind)'*M*q(:,tind)));
  q(:,tind+1) = invMm*(Mp*q(:,tind));
end
E(tind+1) = real(diag(q(:,tind+1)'*M*q(:,tind+1)));

%
% Plot spatio-temporal evolution of disturbance (Fig 6)
%
set(0,'Units','Centimeters');
fig1=figure('Units','Centimeters','Position',[1 1 8 10]);
axes1 = axes('FontName','arial', 'FontSize',6, 'Parent',fig1);
subplot(8,1,[1:6]);   MM=70;
surf(t,xh(MM:end-MM),real(q(MM:end-MM,:)));
ccolor;  hold on;

%
% Find and plot branch I and II
%
xlim([0 tmax]);
dd(1)=find(xh>x1,1,'first');
dd(2)=find(xh>x2,1,'first');
vvv=dd(1):dd(2);
plot3(t,xh(dd(1))*ones(length(t),1),abs(q(dd(1),:)),'k--','linewidth',1)
plot3(t,xh(dd(2))*ones(length(t),1),abs(q(dd(2),:)),'k--','linewidth',1)
annotation1 = annotation(fig1,'textbox', 'LineStyle','none','Position',[0.17 0.30 0.5 0.5], 'FontName','Arial',  'FontSize',7,  'String',{'(a)'});
view([-0.5 -0.6 8]);  axis tight; grid off;  zlim([0 5]); setlabel('t','x')

%
% Plot energy evolution in time
%
subplot(8,1,[7:8])
semilogy(t,E,'r');
annotation1 = annotation(fig1,'textbox', 'LineStyle','none','Position',[0.17 -0.17 0.5 0.5], 'FontName','Arial',  'FontSize',7,  'String',{'(b)'});
axis([0 tmax 0 4e3]);  setlabel('t','E');
set(gca, 'ytick',[0 1 100 ])
